import { hash, verify } from "@node-rs/argon2";
import { encodeBase32LowerCase } from "@oslojs/encoding";
import { redirect, fail } from "@sveltejs/kit";
import { eq } from "drizzle-orm";
import { a as db, u as user, g as generateSessionToken, c as createSession, s as setSessionTokenCookie } from "../../../../chunks/auth.js";
const load = async (event) => {
  if (event.locals.user) {
    return redirect(302, "/priv00/login");
  }
  return {};
};
const actions = {
  login: async (event) => {
    const formData = await event.request.formData();
    const username = formData.get("username");
    const password = formData.get("password");
    if (!validateUsername(username)) {
      return fail(400, {
        message: "Invalid username (min 3, max 31 characters, alphanumeric only)"
      });
    }
    if (!validatePassword(password)) {
      return fail(400, { message: "Invalid password (min 6, max 255 characters)" });
    }
    const results = await db.select().from(user).where(eq(user.username, username));
    const existingUser = results.at(0);
    if (!existingUser) {
      return fail(400, { message: "Incorrect username or password" });
    }
    const validPassword = await verify(existingUser.passwordHash, password, {
      memoryCost: 19456,
      timeCost: 2,
      outputLen: 32,
      parallelism: 1
    });
    if (!validPassword) {
      return fail(400, { message: "Incorrect username or password" });
    }
    const sessionToken = generateSessionToken();
    const session = await createSession(sessionToken, existingUser.id);
    setSessionTokenCookie(event, sessionToken, session.expiresAt);
    return redirect(302, "/priv00");
  },
  register: async (event) => {
    const formData = await event.request.formData();
    const username = formData.get("username");
    const password = formData.get("password");
    if (!validateUsername(username)) {
      return fail(400, { message: "Invalid username" });
    }
    if (!validatePassword(password)) {
      return fail(400, { message: "Invalid password" });
    }
    const userId = generateUserId();
    const passwordHash = await hash(password, {
      // recommended minimum parameters
      memoryCost: 19456,
      timeCost: 2,
      outputLen: 32,
      parallelism: 1
    });
    try {
      await db.insert(user).values({ id: userId, username, passwordHash });
      const sessionToken = generateSessionToken();
      const session = await createSession(sessionToken, userId);
      setSessionTokenCookie(event, sessionToken, session.expiresAt);
    } catch {
      return fail(500, { message: "An error has occurred" });
    }
    return redirect(302, "/priv00");
  }
};
function generateUserId() {
  const bytes = crypto.getRandomValues(new Uint8Array(15));
  const id = encodeBase32LowerCase(bytes);
  return id;
}
function validateUsername(username) {
  return typeof username === "string" && username.length >= 3 && username.length <= 31 && /^[a-z0-9_-]+$/.test(username);
}
function validatePassword(password) {
  return typeof password === "string" && password.length >= 6 && password.length <= 255;
}
export {
  actions,
  load
};
